<?php
/*************************************************
 * NCAAB (Men's) — ESPN Scoreboard -> sports_list
 * Update window:
 *   - 10 minutes before tip, or while games are live/halftime
 *   - If ALL games are GAME_DONE, do NOT update (unless UPDATE_ANYWAY=true)
 * Force update:
 *   - /NCAAB_Update.php?UPDATE_ANYWAY=true
 *************************************************/
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once '../../../config.php'; // adjust path if needed

// ---------- DB ----------
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) die("Connection failed: " . mysqli_connect_error());

// Time (ET)
date_default_timezone_set('America/New_York');
$time_updated      = date('m/d/Y | g:iA');
$current_timestamp = time(); // unix
$current_timestamp_mysql = date('Y-m-d H:i:s', $current_timestamp);

// Query param to force update
$UPDATE_ANYWAY = isset($_GET['UPDATE_ANYWAY']) && strtolower($_GET['UPDATE_ANYWAY']) === 'true';

// Touch API_DB timestamp regardless (so you know the script ran)
mysqli_query($con, "UPDATE `API_DB` SET `TIME_UPDATED`='".mysqli_real_escape_string($con,$time_updated)."' WHERE `API`='NCAAB'");

// Helpers
function hexToRgbCsv($hex) {
    $hex = ltrim($hex ?? '', '#');
    if (strlen($hex) !== 6) return '255,255,255';
    $r = hexdec(substr($hex,0,2));
    $g = hexdec(substr($hex,2,2));
    $b = hexdec(substr($hex,4,2));
    if ($r <= 16 && $g <= 16 && $b <= 16) { $r = $g = $b = 255; } // force white if too dark
    return "{$r},{$g},{$b}";
}
function s($con,$v){ return mysqli_real_escape_string($con, $v ?? ''); }

// ESPN endpoint (no date windows)
$api_url = "https://site.api.espn.com/apis/site/v2/sports/basketball/mens-college-basketball/scoreboard";

// cURL
$ch = curl_init();
curl_setopt_array($ch, [
  CURLOPT_URL => $api_url,
  CURLOPT_RETURNTRANSFER => true,
  CURLOPT_FOLLOWLOCATION => true,
  CURLOPT_TIMEOUT => 30,
  CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
]);
$response = curl_exec($ch);
$err      = curl_error($ch);
curl_close($ch);

$writtenRows = []; // for HTML table
$reasons = [];

if ($err) {
  header('Content-Type: text/plain');
  echo "cURL Error: $err\n";
  exit;
}

$data = json_decode($response, true);
$events = $data['events'] ?? [];

// --------- Decide if we should update (gate) ---------
$any_pre_starting_soon = false;
$any_live_or_ht = false;
$all_done = !empty($events); // assume true; prove otherwise
$first_tip_ts = null;
$last_end_ts  = null;

// Walk once to compute timing & states
foreach ($events as $event) {
    $comp   = $event['competitions'][0] ?? [];
    $status = $comp['status'] ?? [];
    $state  = $status['type']['state'] ?? ''; // pre, in, post

    // Tip time (UTC in API) -> unix
    $game_date_iso = $event['date'] ?? '';
    $tip_ts = $game_date_iso ? strtotime($game_date_iso) : null;

    if ($tip_ts !== null) {
        if ($first_tip_ts === null || $tip_ts < $first_tip_ts) $first_tip_ts = $tip_ts;
        // crude "end time" proxy:
        // if game is post, assume it ended at tip + 3h (college games ~2h-2.5h; buffer)
        // if game is in/ht, treat current time as "not ended yet"
        if ($state === 'post') {
            $end = $tip_ts + 3 * 3600;
            if ($last_end_ts === null || $end > $last_end_ts) $last_end_ts = $end;
        }
    }

    if ($state !== 'post') $all_done = false;

    if ($state === 'in') {
        $any_live_or_ht = true;
    } elseif ($state === 'pre') {
        // 10-minute window
        if ($tip_ts !== null && $tip_ts - $current_timestamp <= 600 && $tip_ts - $current_timestamp >= 0) {
            $any_pre_starting_soon = true;
        }
    } elseif ($state === 'post') {
        // handled via all_done + last_end_ts logic; per your ask, if ALL are done, do not update
    }
}

// Update gate logic
$should_update = false;
if ($UPDATE_ANYWAY) {
    $should_update = true;
    $reasons[] = "Forced (UPDATE_ANYWAY=true)";
} else {
    if ($any_live_or_ht) {
        $should_update = true;
        $reasons[] = "Live/Halftime";
    } elseif ($any_pre_starting_soon) {
        $should_update = true;
        $reasons[] = "Starting within 10 minutes";
    } elseif ($all_done) {
        // per your instruction: if ALL are GAME_DONE, do not update
        $should_update = false;
        $reasons[] = "All games done — skipping";
    } else {
        // Nothing soon and nothing live -> skip
        $should_update = false;
        $reasons[] = "No triggers (not live/HT and not starting within 10m)";
    }
}

// If we shouldn't update, render a nice status page and exit
if (!$should_update) {
    ?>
    <!DOCTYPE html>
    <html>
    <head>
      <meta charset="utf-8" />
      <title>NCAAB Update — Skipped</title>
      <style>
        body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu; margin:0; padding:0; background:#0f172a; color:#e2e8f0; min-height:100vh; display:flex; align-items:center; justify-content:center;}
        .box{background:#0b1022; border:1px solid #1e293b; padding:28px; border-radius:14px; width:min(880px,92vw)}
        h1{margin:0 0 6px}
        .muted{opacity:.8}
        ul{margin:10px 0 0 18px}
        .badge{display:inline-block; background:#1e293b; padding:4px 8px; border-radius:999px; font-size:12px}
      </style>
    </head>
    <body>
      <div class="box">
        <h1>🏀 NCAAB Update Skipped</h1>
        <div class="muted">Last check: <?= htmlspecialchars(date('m/d/Y | g:iA')) ?> ET</div>
        <p>Update conditions not met.</p>
        <ul>
          <?php foreach ($reasons as $r): ?>
            <li><?= htmlspecialchars($r) ?></li>
          <?php endforeach; ?>
        </ul>
        <p class="muted" style="margin-top:10px">Tip: append <code>?UPDATE_ANYWAY=true</code> to force a write.</p>
      </div>
    </body>
    </html>
    <?php
    mysqli_close($con);
    exit;
}

// ---------------- Perform the update ----------------
function teamBlock($comp) {
    $t = $comp['team'] ?? [];
    $color = $t['color'] ?? '';
    if (($color ?? '') === '000000' && !empty($t['alternateColor'])) $color = $t['alternateColor'];
    return [
        'id'        => strval($t['id'] ?? ''),
        'name'      => strval($t['name'] ?? ''),
        'location'  => strval($t['location'] ?? ''),
        'abbrev'    => strval($t['abbreviation'] ?? ''),
        'score'     => strval($comp['score'] ?? '0'),
        'color_rgb' => hexToRgbCsv($color),
        'record'    => (function($comp){
            $out = '';
            foreach (($comp['records'] ?? []) as $r) {
                $key = $r['type'] ?? $r['name'] ?? '';
                if ($key === 'overall' || $key === 'total') { $out = $r['summary'] ?? ''; break; }
            }
            if ($out === '' && isset($comp['records'][0]['summary'])) $out = $comp['records'][0]['summary'];
            return $out;
        })($comp),
    ];
}

$gameNum = 0;
foreach ($events as $event) {
    $gameNum++;

    $competition = $event['competitions'][0] ?? [];
    $status      = $competition['status'] ?? [];
    $state       = $status['type']['state'] ?? '';   // pre, in, post
    $period      = intval($status['period'] ?? 0);
    $clock       = $status['displayClock'] ?? '0:00';

    // Map game state to our values
    $game_status = 'NOT_STARTED';
    if ($state === 'in') {
        $game_status = ($period === 2 && $clock === '0:00') ? 'HALF_TIME' : 'IN_SESSION';
    } elseif ($state === 'post') {
        $game_status = 'GAME_DONE';
    }

    $game_date_iso = $event['date'] ?? '';
    $game_time_et  = $game_date_iso ? date("g:i a", strtotime($game_date_iso)) : '';

    // Broadcasts
    $broadcast = '';
    if (!empty($competition['broadcasts']) && is_array($competition['broadcasts'])) {
        $names = [];
        foreach ($competition['broadcasts'] as $b) {
            if (!empty($b['names']) && is_array($b['names'])) {
                $names = array_merge($names, $b['names']);
            }
        }
        if (!empty($names)) $broadcast = implode(', ', array_slice(array_unique($names), 0, 2));
    }

    // Teams
    $home = []; $away = [];
    foreach (($competition['competitors'] ?? []) as $comp) {
        if (($comp['homeAway'] ?? '') === 'home') $home = teamBlock($comp);
        if (($comp['homeAway'] ?? '') === 'away') $away = teamBlock($comp);
    }

    // Situation
    $who_has = '';
    $lastPlayText = '';
    $scoreValueFlag = 'NO-SCORE';
    if (in_array($game_status, ['IN_SESSION','HALF_TIME'], true)) {
        $sit = $competition['situation'] ?? [];
        $lastPlay = $sit['lastPlay'] ?? [];
        $lastPlayText = strval($lastPlay['text'] ?? '');
        $sv = intval($lastPlay['scoreValue'] ?? 0);
        if ($sv > 0) $scoreValueFlag = 'SCORE';
        $possId = $sit['possession'] ?? '';
        if ($possId && $possId === ($home['id'] ?? '')) $who_has = $home['abbrev'];
        if ($possId && $possId === ($away['id'] ?? '')) $who_has = $away['abbrev'];
    }

    // Odds
    $favoriteTeam = '';
    $favoritePoints = '';
    $overUnder = '';
    if (!empty($competition['odds'][0])) {
        $odds = $competition['odds'][0];
        $overUnder = strval($odds['overUnder'] ?? '');
        $details   = strval($odds['details'] ?? '');
        if (!empty($home['abbrev']) && strpos($details, $home['abbrev']) !== false) $favoriteTeam = $home['abbrev'];
        if (!empty($away['abbrev']) && strpos($details, $away['abbrev']) !== false) $favoriteTeam = $away['abbrev'];
        if ($details !== '' && preg_match('/[-+]?(\d+(\.\d+)?)/', $details, $m)) $favoritePoints = $m[0];
    }

    // Colors split
    $rgb1 = array_map('intval', explode(',', $home['color_rgb'] ?? '255,255,255'));
    $rgb2 = array_map('intval', explode(',', $away['color_rgb'] ?? '255,255,255'));

    // Build update row
    $row = [
        'Clock'            => $clock,
        'Period'           => $period,
        'Game_Played'      => $game_status,
        'Game_Time'        => $game_time_et,
        'Game_Date'        => $game_date_iso,
        'broadcast'        => $broadcast,
        'scoreValue'       => $scoreValueFlag,
        'Team_1_Name'      => $home['name'] ?? '',
        'Team_2_Name'      => $away['name'] ?? '',
        'Team_1_Record'    => $home['record'] ?? '',
        'Team_2_Record'    => $away['record'] ?? '',
        'Team_1_Score'     => intval($home['score'] ?? 0),
        'Team_2_Score'     => intval($away['score'] ?? 0),
        'Team_1_Location'  => $home['location'] ?? '',
        'Team_2_Location'  => $away['location'] ?? '',
        'Team_1_Abbrev'    => $home['abbrev'] ?? '',
        'Team_2_Abbrev'    => $away['abbrev'] ?? '',
        'Team_1_Color'     => '"'.($home['color_rgb'] ?? '255,255,255').'"',
        'Team_2_Color'     => '"'.($away['color_rgb'] ?? '255,255,255').'"',
        'Team_1_Color_R'   => $rgb1[0] ?? 255,
        'Team_1_Color_G'   => $rgb1[1] ?? 255,
        'Team_1_Color_B'   => $rgb1[2] ?? 255,
        'Team_2_Color_R'   => $rgb2[0] ?? 255,
        'Team_2_Color_G'   => $rgb2[1] ?? 255,
        'Team_2_Color_B'   => $rgb2[2] ?? 255,
        'Who_Has_Ball'     => $who_has,
        'lastPlay'         => $lastPlayText,
        'OverUnder'        => $overUnder,
        'Favorite_Team'    => $favoriteTeam,
        'Favorite_Points'  => $favoritePoints,
        'lastUpdatedTime'  => $current_timestamp_mysql,
    ];

    // Write
    $sets = [];
    foreach ($row as $field => $val) {
        if ($field === 'Team_1_Color' || $field === 'Team_2_Color') {
            $sets[] = "$field=$val"; // already quoted "r,g,b"
        } else {
            $sets[] = "$field='".s($con, (string)$val)."'";
        }
    }
    $sql = "UPDATE `sports_list` SET ".implode(', ',$sets)." WHERE `Game`={$gameNum} AND `Conference`='NCAAB'";
    mysqli_query($con, $sql);

    // For HTML
    $writtenRows[] = [
        'Game'           => $gameNum,
        'Status'         => $row['Game_Played'],
        'Time'           => $row['Game_Time'],
        'Date'           => $row['Game_Date'],
        'Broadcast'      => $row['broadcast'],
        'Away'           => $row['Team_2_Abbrev'],
        'AwayScore'      => $row['Team_2_Score'],
        'AwayRec'        => $row['Team_2_Record'],
        'Home'           => $row['Team_1_Abbrev'],
        'HomeScore'      => $row['Team_1_Score'],
        'HomeRec'        => $row['Team_1_Record'],
        'Period'         => $row['Period'],
        'Clock'          => $row['Clock'],
        'ScoreEvent'     => $row['scoreValue'],
        'LastPlay'       => $row['lastPlay'],
        'Possession'     => $row['Who_Has_Ball'],
        'Favorite'       => $row['Favorite_Team'],
        'Spread'         => $row['Favorite_Points'],
        'OU'             => $row['OverUnder'],
        'Updated'        => $row['lastUpdatedTime'],
    ];
}

// ---------- HTML table ----------
?>
<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8" />
  <title>NCAAB Database Update</title>
  <style>
    body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu; margin:20px; background:#0f172a; color:#fff;}
    h1,h2{margin:0 0 10px}
    .muted{opacity:.8}
    .card{background:#0b1022; border:1px solid #1e293b; border-radius:12px; padding:16px; margin-top:14px;}
    table{width:100%; border-collapse:collapse; font-size:12px; background:#0b1226}
    th,td{padding:8px 6px; border-bottom:1px solid #1e293b; vertical-align:top; color:#cbd5e1}
    th{position:sticky; top:0; background:#0b1022; text-transform:uppercase; letter-spacing:.4px; font-weight:700; color:#93c5fd}
    .status-IN_SESSION{color:#22c55e; font-weight:700}
    .status-HALF_TIME{color:#f59e0b; font-weight:700}
    .status-GAME_DONE{color:#94a3b8; font-weight:700}
    .status-NOT_STARTED{color:#64748b; font-weight:700}
    .chip{display:inline-block; background:#1e293b; padding:2px 6px; border-radius:10px; font-size:11px}
    .bar{display:flex; gap:10px; align-items:center}
    .tag{background:#1e293b; padding:4px 8px; border-radius:999px; font-size:12px}
  </style>
</head>
<body>
  <div class="bar">
    <h1>🏀 NCAAB Games — Database Update</h1>
    <?php foreach ($reasons as $r): ?>
      <span class="tag"><?= htmlspecialchars($r) ?></span>
    <?php endforeach; ?>
  </div>
  <div class="muted">Last Updated: <?= htmlspecialchars($time_updated) ?> ET &middot; Wrote <?= count($writtenRows) ?> game(s)<?php if($UPDATE_ANYWAY) echo " (forced)"; ?></div>

  <div class="card">
    <table>
      <thead>
        <tr>
          <th>Game</th>
          <th>Status</th>
          <th>Tip</th>
          <th>Date (UTC)</th>
          <th>TV</th>
          <th>Away</th>
          <th>Away&nbsp;Score</th>
          <th>Away&nbsp;Rec</th>
          <th>Home</th>
          <th>Home&nbsp;Score</th>
          <th>Home&nbsp;Rec</th>
          <th>Per</th>
          <th>Clock</th>
          <th>Score</th>
          <th>Last Play</th>
          <th>Poss</th>
          <th>Fav</th>
          <th>Spread</th>
          <th>O/U</th>
          <th>Wrote</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($writtenRows as $r): ?>
        <tr>
          <td><?= intval($r['Game']) ?></td>
          <td class="status-<?= htmlspecialchars($r['Status']) ?>"><?= htmlspecialchars($r['Status']) ?></td>
          <td><?= htmlspecialchars($r['Time']) ?></td>
          <td><?= htmlspecialchars($r['Date']) ?></td>
          <td><?= htmlspecialchars($r['Broadcast'] ?: '-') ?></td>
          <td><strong><?= htmlspecialchars($r['Away']) ?></strong></td>
          <td><strong><?= intval($r['AwayScore']) ?></strong></td>
          <td><?= htmlspecialchars($r['AwayRec']) ?></td>
          <td><strong><?= htmlspecialchars($r['Home']) ?></strong></td>
          <td><strong><?= intval($r['HomeScore']) ?></strong></td>
          <td><?= htmlspecialchars($r['HomeRec']) ?></td>
          <td><?= htmlspecialchars((string)$r['Period']) ?></td>
          <td><?= htmlspecialchars($r['Clock']) ?></td>
          <td><?= htmlspecialchars($r['ScoreEvent']) ?></td>
          <td title="<?= htmlspecialchars($r['LastPlay']) ?>">
            <?= $r['LastPlay'] ? htmlspecialchars(mb_strimwidth($r['LastPlay'],0,60,'…')) : '-' ?>
          </td>
          <td><?= htmlspecialchars($r['Possession'] ?: '-') ?></td>
          <td><?= htmlspecialchars($r['Favorite'] ?: '-') ?></td>
          <td><?= htmlspecialchars($r['Spread'] ?: '-') ?></td>
          <td><?= htmlspecialchars($r['OU'] ?: '-') ?></td>
          <td><span class="chip"><?= htmlspecialchars($r['Updated']) ?></span></td>
        </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>
</body>
</html>
<?php
mysqli_close($con);
